<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Http\JsonResponse;
use App\Models\Service;
use Illuminate\Support\Str;

class ServiceTagController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $data = [];

            $services = Service::where('is_active', 1)->get();
            if ($services->isEmpty()) throw new Exception('No data found.', 404);

            foreach ($services as $service) {
                $tags = explode(', ', $service->tags);
                $data = array_merge($data, $tags);
            }

            $uniqueData = array_unique($data);
            $tagCounts = array_count_values($data);

            $result = [
                [
                    'tag' => 'All',
                    'slug' => 'all',
                    'count' => count($services),
                ]
            ];
            foreach ($uniqueData as $tag) {
                $result[] = [
                    'tag' => $tag,
                    'slug' => strtolower(str_replace(' ', '-', $tag)),
                    'count' => $tagCounts[$tag],
                ];
            }

            return response()->json($result, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }


    /**
     * Display the specified resource.
     */
    public function show(String $tag, Request $request): JsonResponse
    {
        try {
            $tag = str_replace('-', ' ', $tag);
            $perPage = $request->input('per_page', 10);

            $query = Service::select(
                'id',
                'category_id',
                'image',
                'name',
                'service_type',
                'price',
                'discounted_price',
                'label_id',
                'tags'
            )
                ->with([
                    'label' => function ($query) {
                        $query->select('id', 'name');
                    },
                    'category' => function ($query) {
                        $query->select('id', 'name')->where('is_active', 1);
                    }
                ])
                ->where('is_active', 1);

            if ($tag !== 'all') {
                $query->whereRaw('LOWER(tags) LIKE ?', ['%' . strtolower($tag) . '%']);
            }

            $services = $query->paginate($perPage);

            if ($services->isEmpty()) {
                return response()->json(['error' => 'No services found for the provided tag'], 404);
            }

            return response()->json($services, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
