<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\GiftOrder;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Models\Order;
use App\Models\OrderHasService;
use App\Models\Service;
use App\Models\User;
use Illuminate\Support\Facades\Auth;

class OrderController extends Controller
{
    protected $user;

    public function __construct()
    {
        $this->user = Auth::user();
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $perPage = $request->query('per_page', 10);
            $data = Order::with('services')->where('user_id', $this->user->id)->orderBy('id', 'desc')->paginate($perPage);

            if (empty($data)) throw new Exception('No data found', 404);
            return response()->json($data, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(String $id): JsonResponse
    {
        try {
            $data = Order::with('services')->where('user_id', $this->user->id)->where('id', $id)->first();
            if (empty($data)) throw new Exception('No data found', 404);

            return response()->json($data, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            DB::beginTransaction();

            $validator = Validator::make(
                $request->all(),
                [
                    'gift_order_id' => 'nullable|exists:gift_orders,id',
                    'gift_amount' => 'nullable|numeric|required_with:gift_order_id',
                    'user_name' => 'required|max:255',
                    'user_email' => 'required|max:255',
                    'user_phone' => 'required|max:255',
                    'services_amount' => 'required|numeric',
                    'total_amount' => 'required|numeric',
                    'payment_method' => 'nullable|max:255',
                    'transaction_id' => 'nullable|max:255',
                    'transaction_card_name' => 'nullable|max:255',
                    'transaction_card_number' => 'nullable|max:255',
                    'transaction_cvc' => 'nullable|max:255',
                    'transaction_expiry_year' => 'nullable|date_format:Y',
                    'transaction_expiry_month' => 'nullable|date_format:m',
                ],
                [
                    'gift_order_id.exists' => 'Invalid Gift.',

                    'gift_amount.required_with' => 'Gift amount required.',
                    'gift_amount.numeric' => 'Gift amount numeric.',

                    'user_name.required' => 'Name required.',
                    'user_name.max' => 'Name must be less then 255 characters.',

                    'user_email.required' => 'Email required.',
                    'user_email.max' => 'Email must be less then 255 characters.',

                    'user_phone.required' => 'Phone required.',
                    'user_phone.max' => 'Phone must be less then 255 characters.',

                    'services_amount.required' => 'Services amount required.',
                    'services_amount.numeric' => 'Services amount numeric.',

                    'total_amount.required' => 'Total amount required.',
                    'total_amount.numeric' => 'Total amount numeric.',

                    'payment_method.required' => 'Payment method required.',
                    'payment_method.max' => 'Payment method must be less then 255 characters.',

                    'transaction_id.required' => 'Transaction id required.',
                    'transaction_id.max' => 'Transaction id must be less then 255 characters.',

                    'transaction_card_name.required' => 'Card name required.',
                    'transaction_card_name.max' => 'Card name must be less then 255 characters.',

                    'transaction_card_number.required' => 'Card number required.',
                    'transaction_card_number.max' => 'Card number must be less then 255 characters.',

                    'transaction_cvc.required' => 'Cvc required.',
                    'transaction_cvc.max' => 'Cvc must be less then 255 characters.',

                    'transaction_expiry_year.required' => 'Expiry year required.',
                    'transaction_expiry_year.date_format' => 'Expiry year invalid.',

                    'transaction_expiry_month.required' => 'Expiry month required.',
                    'transaction_expiry_month.date_format' => 'Expiry month invalid.',
                ]
            );

            if ($validator->fails()) throw new Exception($validator->errors()->first(), 400);

            $user = User::where('id', $this->user->id)->first();

            $data = new Order();

            $data->user_id = $this->user->id;

            if ($request->gift_order_id != null) {
                $data->gift_order_id = $request->gift_order_id;
                $data->gift_amount = $request->gift_amount;
            }

            $data->user_name = $request->user_name;
            $data->user_email = $request->user_email;
            $data->user_phone = $request->user_phone;
            $data->services_amount = $request->services_amount;
            $data->total_amount = $request->total_amount;
            $data->payment_method = $request->payment_method;
            $data->transaction_id = $request->transaction_id;
            $data->transaction_card_name = $request->transaction_card_name;
            $data->transaction_card_number = $request->transaction_card_number;
            $data->transaction_cvc = $request->transaction_cvc;
            $data->transaction_expiry_year = $request->transaction_expiry_year;
            $data->transaction_expiry_month = $request->transaction_expiry_month;
            $data->save();

            foreach ($request->services as $service) {
                $serviceModal = Service::where('id', $service['id'])->first();

                $serviceData = new OrderHasService();
                $serviceData->order_id = $data->id;
                $serviceData->service_id = $serviceModal['id'];
                $serviceData->image = $serviceModal->image;
                $serviceData->name = $serviceModal->name;
                $serviceData->service_type = $service['type'];
                $serviceData->qty = $service['qty'];
                $serviceData->price = $service['price'];
                $serviceData->total_price = $service['total_price'];
                $serviceData->save();
            }

            if ($request->gift_order_id != null) {
                $gift = GiftOrder::find($request->gift_order_id)->first();
                $gift->balance = $gift->balance - $request->gift_amount;
                $gift->save();
            }

            DB::commit();
            return response()->json($data, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
