<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Models\Service;

class ServiceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $perPage = $request->query('per_page', 10);
            $type = $request->query('type');

            $data = Service::with('category')->with('label')->where('parent_id', 0)->orderBy('id', 'desc');
            if ($type != null && $type == 'one_time') $data = $data->whereNotNull('price');
            if ($type != null && $type == 'monthly') $data = $data->whereNotNull('monthly_price');
            $data = $data->paginate($perPage);

            if (empty($data)) throw new Exception('No data found', 404);
            return response()->json($data, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(String $id): JsonResponse
    {
        try {
            $data = Service::where('id', $id)->with('category')->with('label')->first();
            if (empty($data)) throw new Exception('No data found', 404);

            return response()->json($data, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            DB::beginTransaction();

            $validator = Validator::make(
                $request->all(),
                [
                    'category_id' => 'required|exists:categories,id',
                    'label_id' => 'required|exists:labels,id',
                    'name' => 'required|max:255',
                    'image' => 'required|file|mimetypes:image/*',
                    'price' => 'nullable|numeric|required_if:monthly_price,null',
                    'discounted_price' => 'nullable|numeric',
                    'monthly_price' => 'nullable|numeric|required_if:price,null',
                    'monthly_discounted_price' => 'nullable|numeric',
                    'detail' => 'required',
                    'brief_detail' => 'nullable',
                    'includes' => 'nullable',
                    'description' => 'nullable',
                    'requirements' => 'nullable',
                    'notes' => 'nullable',
                    'tags' => 'nullable',
                ],
                [
                    'category_id.required' => 'Category required.',
                    'category_id.exists' => 'Invalid category.',

                    'label_id.required' => 'Label required.',
                    'label_id.exists' => 'Invalid label.',

                    'name.required' => 'Name required.',
                    'name.max' => 'Name maximum 255 characters.',

                    'image.required' => 'Image required.',
                    'image.file' => 'Invalid image.',
                    'image.mimetypes' => 'Invalid image format.',

                    'service_type.required' => 'Service type required.',
                    'service_type.in' => 'Service type only one_time or monthly.',

                    'price.required_if' => 'Price required.',
                    'price.numeric' => 'Invalid price.',

                    'discounted_price.required' => 'Discounted price required.',
                    'discounted_price.numeric' => 'Invalid discounted price.',

                    'monthly_price.required_if' => 'Monthly price required.',
                    'monthly_price.numeric' => 'Invalid monthly price.',

                    'detail.required' => 'Detail required.',
                ]
            );

            if ($validator->fails()) throw new Exception($validator->errors()->first(), 400);

            $image = $request->file('image');
            $image_name = 'service_image_' . time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('service-images'), $image_name);

            $parent_id = 0;

            if ($request->price != null) {
                $data = new Service();
                $data->category_id = $request->category_id;
                $data->label_id = $request->label_id;
                $data->name = $request->name;
                $data->image = 'service-images/' . $image_name;
                $data->service_type = 'one_time';
                $data->price = $request->price;
                $data->discounted_price = $request->discounted_price;
                $data->detail = $request->detail;
                $data->brief_detail = $request->brief_detail;
                $data->includes = $request->includes;
                $data->description = $request->description;
                $data->requirements = $request->requirements;
                $data->notes = $request->notes;
                $data->tags = $request->tags;
                $data->save();
                $parent_id = $data->id;
            }

            if ($request->monthly_price != null) {    
                $anotherData = new Service();
                $anotherData->parent_id = $parent_id;
                $anotherData->category_id = $request->category_id;
                $anotherData->label_id = $request->label_id;
                $anotherData->name = $request->name;
                $anotherData->image = 'service-images/' . $image_name;
                $anotherData->service_type = 'subscription';
                $anotherData->price = $request->monthly_price;
                $anotherData->discounted_price = $request->monthly_discounted_price;
                $anotherData->detail = $request->detail;
                $anotherData->brief_detail = $request->brief_detail;
                $anotherData->includes = $request->includes;
                $anotherData->description = $request->description;
                $anotherData->requirements = $request->requirements;
                $anotherData->notes = $request->notes;
                $anotherData->tags = $request->tags;
                $anotherData->save();
            }

            DB::commit();

            return response()->json([
                'status' => 'success',
            ], 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, String $id): JsonResponse
    {
        try {
            $validator = Validator::make(
                $request->all(),
                [
                    'category_id' => 'required|exists:categories,id',
                    'label_id' => 'required|exists:labels,id',
                    'name' => 'required|max:255',
                    'image' => 'nullable|file|mimetypes:image/*',
                    'price' => 'nullable|numeric|required_if:monthly_price,null',
                    'discounted_price' => 'nullable|numeric',
                    'monthly_price' => 'nullable|numeric|required_if:price,null',
                    'monthly_discounted_price' => 'nullable|numeric',
                    'detail' => 'required',
                    'brief_detail' => 'nullable',
                    'includes' => 'nullable',
                    'description' => 'nullable',
                    'requirements' => 'nullable',
                    'notes' => 'nullable',
                    'tags' => 'nullable',
                ],
                [
                    'category_id.required' => 'Category required.',
                    'category_id.exists' => 'Invalid category.',

                    'label_id.required' => 'Label required.',
                    'label_id.exists' => 'Invalid label.',

                    'name.required' => 'Name required.',
                    'name.max' => 'Name maximum 255 characters.',

                    'image.required' => 'Image required.',
                    'image.file' => 'Invalid image.',
                    'image.mimetypes' => 'Invalid image format.',

                    'service_type.required' => 'Service type required.',
                    'service_type.in' => 'Service type only one_time or monthly.',

                    'price.required' => 'Price required.',
                    'price.numeric' => 'Invalid price.',

                    'discounted_price.required' => 'Discounted price required.',
                    'discounted_price.numeric' => 'Invalid discounted price.',

                    'detail.required' => 'Detail required.',
                ]
            );

            if ($validator->fails()) throw new Exception($validator->errors()->first(), 400);

            $data = Service::find($id);
            if (empty($data)) throw new Exception('No data found', 404);

            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $image_name = 'service_image_' . time() . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('service-images'), $image_name);

                unlink(public_path($data->image));
                $data->image = 'service-images/' . $image_name;
            }

            $data->category_id = $request->category_id;
            $data->label_id = $request->label_id;
            $data->name = $request->name;
            $data->service_type = 'one_time';
            $data->price = $request->price;
            $data->discounted_price = $request->discounted_price;
            $data->detail = $request->detail;
            $data->brief_detail = $request->brief_detail;
            $data->includes = $request->includes;
            $data->description = $request->description;
            $data->requirements = $request->requirements;
            $data->notes = $request->notes;
            $data->tags = $request->tags;
            $data->save();

            if ($data->parent_id != null) {
                $childData = Service::where('id', $data->parent_id)->first();
                if (!empty($childData)) {
                    $childData->category_id = $data->category_id;
                    $childData->label_id = $data->label_id;
                    $childData->name = $data->name;
                    $childData->service_type = 'subscription';
                    $childData->price = $request->monthly_price;
                    $childData->discounted_price = $request->monthly_discounted_price;
                    $childData->detail = $data->detail;
                    $childData->brief_detail = $data->brief_detail;
                    $childData->includes = $data->includes;
                    $childData->description = $data->description;
                    $childData->requirements = $data->requirements;
                    $childData->notes = $data->notes;
                    $childData->tags = $data->tags;
                    $childData->save();
                }
            }

            return response()->json([
                'status' => 'success',
            ], 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function updateStatus(Request $request, String $id): JsonResponse
    {
        try {
            $validator = Validator::make(
                $request->all(),
                [
                    'status' => 'required|boolean',
                ],
                [
                    'status.required' => 'Status required.',
                ]
            );

            if ($validator->fails()) throw new Exception($validator->errors()->first(), 400);

            $data = Service::find($id);
            if (empty($data)) throw new Exception('No data found', 404);

            $data->is_active = $request->status;
            $data->save();

            if ($data->parent_id != null) {
                $childData = Service::where('id', $data->parent_id)->first();
                if (!empty($childData)) {
                    $childData->is_active = $data->is_active;
                    $childData->save();
                }
            }

            return response()->json($data, 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(String $id): JsonResponse
    {
        try {
            $data = Service::find($id);
            if (empty($data)) throw new Exception('No data found', 404);
            
            if ($data->parent_id != null) {
                $childData = Service::where('id', $data->parent_id)->first();
                if (!empty($childData)) $childData->delete();
            }

            $data->delete();

            return response()->json('Deleted', 200);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], $e->getCode() ?: 500);
        }
    }
}
